CREATE SEQUENCE S_USER START WITH 1 INCREMENT BY 1;
CREATE TABLE USERS
(
    ID                   INTEGER   DEFAULT NEXTVAL('S_USER') PRIMARY KEY,
    EMAIL                VARCHAR(1000)                         NOT NULL UNIQUE,
    LOGIN                VARCHAR(255) UNIQUE,
    PASSWORD             VARCHAR(100)                          NOT NULL,
    FULL_NAME            VARCHAR(1000)                         NOT NULL,
    IS_ENABLED           BOOLEAN   DEFAULT TRUE                NOT NULL,
    IS_SUBSCRIBED        BOOLEAN   DEFAULT TRUE                NOT NULL,
    IS_AUTO_CREATED      BOOLEAN   DEFAULT FALSE               NOT NULL,
    PASSWORD_CHANGE_DATE TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    LOCKED_DATE          TIMESTAMP,
    FAILED_LOGIN_COUNT   INTEGER                               NOT NULL DEFAULT 0,
    DESCRIPTION          VARCHAR(125000),
    REGISTERED_DATE      TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL
);

CREATE SEQUENCE S_USER_PASSWORD_OLD START WITH 1 INCREMENT BY 1;
CREATE TABLE USER_PASSWORD_OLD
(
    ID          INTEGER   DEFAULT NEXTVAL('S_USER_PASSWORD_OLD') PRIMARY KEY,
    USER_ID     INTEGER                               NOT NULL,
    PASSWORD    VARCHAR(100)                          NOT NULL,
    DATE_CREATE TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    CONSTRAINT USER_PASSWORD_OLD_USER_FK
        FOREIGN KEY (USER_ID) REFERENCES USERS
            ON UPDATE CASCADE
);

CREATE SEQUENCE S_USER_VISIT START WITH 1 INCREMENT BY 1;
CREATE TABLE USER_VISIT
(
    ID           BIGINT    DEFAULT NEXTVAL('S_USER_VISIT') PRIMARY KEY,
    USER_ID      INTEGER                               NOT NULL,
    IP           VARCHAR(30)                           NOT NULL,
    VISITED_DATE TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    CONSTRAINT USER_VISIT_USER_USER_ID_FK
        FOREIGN KEY (USER_ID) REFERENCES USERS
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_ROLE START WITH 10 INCREMENT BY 1;
CREATE TABLE ROLE
(
    ID          INTEGER DEFAULT NEXTVAL('S_ROLE') PRIMARY KEY,
    NAME        VARCHAR(20)           NOT NULL UNIQUE,
    DESCRIPTION VARCHAR(15000),
    READ_ONLY   BOOLEAN DEFAULT FALSE NOT NULL
);

CREATE TABLE RULE
(
    ROLE_ID INTEGER      NOT NULL,
    NAME    VARCHAR(200) NOT NULL,
    PRIMARY KEY (ROLE_ID, NAME),
    CONSTRAINT RULE_RULE_ROLE_ID_FK
        FOREIGN KEY (ROLE_ID) REFERENCES ROLE
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE ROLE_USER
(
    USER_ID INTEGER NOT NULL,
    ROLE_ID INTEGER NOT NULL,
    PRIMARY KEY (USER_ID, ROLE_ID),
    CONSTRAINT ROLE_USER_USER_USER_ID_FK
        FOREIGN KEY (USER_ID) REFERENCES USERS
            ON UPDATE CASCADE ON DELETE CASCADE,
    CONSTRAINT ROLE_USER_ROLE_ROLE_ID_FK
        FOREIGN KEY (ROLE_ID) REFERENCES ROLE
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_AUDIT START WITH 1 INCREMENT BY 1;
CREATE TABLE AUDIT
(
    ID          BIGINT    DEFAULT NEXTVAL('S_AUDIT') PRIMARY KEY,
    DATE_CREATE TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    USER_ID     INTEGER                               NOT NULL,
    ACTION      VARCHAR(10)                           NOT NULL,
    ENTITY      VARCHAR(200)                          NOT NULL,
    JSON        VARCHAR(200000),
    DESCRIPTION VARCHAR(15000),
    IS_ERROR    BOOLEAN   DEFAULT FALSE               NOT NULL,
    CONSTRAINT AUDIT_USER_USER_ID_FK
        FOREIGN KEY (USER_ID) REFERENCES USERS
            ON UPDATE CASCADE
);

CREATE SEQUENCE S_SERVER START WITH 10 INCREMENT BY 1;
CREATE TABLE SERVER
(
    ID          INTEGER DEFAULT NEXTVAL('S_SERVER') PRIMARY KEY,
    TITLE       VARCHAR(255)          NOT NULL UNIQUE,
    IS_GROUP    BOOLEAN DEFAULT FALSE NOT NULL,
    IS_ACTIVE   BOOLEAN DEFAULT TRUE  NOT NULL,
    DESCRIPTION VARCHAR(125000)
);

CREATE TABLE SERVER_CONFIG
(
    SERVER_ID     INTEGER PRIMARY KEY,
    HOST          VARCHAR(100) NOT NULL,
    LAUNCHER_PORT SMALLINT     NOT NULL,
    RUNTIME_PORT  SMALLINT     NOT NULL,
    CHARSET       VARCHAR(20),
    UNIQUE (HOST, LAUNCHER_PORT),
    UNIQUE (HOST, RUNTIME_PORT),
    CONSTRAINT SERVER_PARAM_SERVER_SERVER_ID_FK FOREIGN KEY (SERVER_ID) REFERENCES SERVER
        ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE SERVER_PROPERTY
(
    SERVER_ID      INTEGER      NOT NULL,
    PROPERTY_NAME  VARCHAR(255) NOT NULL,
    PROPERTY_VALUE VARCHAR(1000),
    PRIMARY KEY (SERVER_ID, PROPERTY_NAME),
    CONSTRAINT SERVER_PROPERTY_SERVER_SERVER_ID_FK FOREIGN KEY (SERVER_ID) REFERENCES SERVER
        ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_SERVER_VARIABLE START WITH 1 INCREMENT BY 1;
CREATE TABLE SERVER_VARIABLE
(
    ID             BIGINT  DEFAULT NEXTVAL('S_SERVER_VARIABLE') PRIMARY KEY,
    SERVER_ID      INTEGER              NOT NULL,
    VARIABLE_NAME  VARCHAR(300)         NOT NULL,
    VARIABLE_VALUE VARCHAR(1000),
    VARIABLE_TYPE  VARCHAR(20)          NOT NULL,
    IS_ACTIVE      BOOLEAN DEFAULT TRUE NOT NULL,
    UNIQUE (SERVER_ID, VARIABLE_NAME),
    CONSTRAINT SERVER_VARIABLE_SERVER_SERVER_ID_FK FOREIGN KEY (SERVER_ID) REFERENCES SERVER
        ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE SERVER_GROUP
(
    SERVER_GROUP_ID INTEGER NOT NULL,
    SERVER_ID       INTEGER NOT NULL,
    PRIMARY KEY (SERVER_GROUP_ID, SERVER_ID),
    CONSTRAINT SERVER_GROUP_SERVER_SERVER_GROUP_ID_FK
        FOREIGN KEY (SERVER_GROUP_ID) REFERENCES SERVER
            ON UPDATE CASCADE ON DELETE CASCADE,
    CONSTRAINT SERVER_GROUP_SERVER_SERVER_ID_FK
        FOREIGN KEY (SERVER_ID) REFERENCES SERVER
            ON UPDATE CASCADE ON DELETE CASCADE
);


CREATE SEQUENCE S_PROJECT START WITH 100 INCREMENT BY 1;
CREATE TABLE PROJECT
(
    ID          INTEGER DEFAULT NEXTVAL('S_PROJECT') PRIMARY KEY,
    TITLE       VARCHAR(500) UNIQUE NOT NULL,
    DESCRIPTION VARCHAR(125000)
);

CREATE TABLE PROJECT_ACCESS
(
    PROJECT_ID     INTEGER NOT NULL,
    ROLE_ID        INTEGER NOT NULL,
    ROLE_READ      BOOLEAN DEFAULT FALSE,
    ROLE_WRITE     BOOLEAN DEFAULT FALSE,
    ROLE_DELETE    BOOLEAN DEFAULT FALSE,
    ROLE_RUN       BOOLEAN DEFAULT FALSE,
    ROLE_SUBSCRIBE BOOLEAN DEFAULT FALSE,
    PRIMARY KEY (PROJECT_ID, ROLE_ID),
    CONSTRAINT PROJECT_ACCESS_PROJECT_PROJECT_ID_FK
        FOREIGN KEY (PROJECT_ID) REFERENCES PROJECT
            ON UPDATE CASCADE ON DELETE CASCADE,
    CONSTRAINT PROJECT_ACCESS_ROLE_ROLE_ID_FK
        FOREIGN KEY (ROLE_ID) REFERENCES ROLE
            ON UPDATE CASCADE ON DELETE CASCADE
);


CREATE SEQUENCE S_SUBSCRIBE_MESSAGE START WITH 1 INCREMENT BY 1;
CREATE TABLE SUBSCRIBE_MESSAGE
(
    ID          BIGINT    DEFAULT NEXTVAL('S_SUBSCRIBE_MESSAGE') PRIMARY KEY,
    PROJECT_ID  INTEGER,
    TYPE        VARCHAR(20)                           NOT NULL,
    SUBJECT     VARCHAR(1000)                         NOT NULL,
    BODY        VARCHAR(125000),
    IS_SEND     BOOLEAN   DEFAULT FALSE               NOT NULL,
    DATE_CREATE TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    CONSTRAINT SUBSCRIBE_MESSAGE_PROJECT_FK
        FOREIGN KEY (PROJECT_ID) REFERENCES PROJECT
            ON UPDATE CASCADE ON DELETE SET NULL
);

CREATE SEQUENCE S_PACKAGE START WITH 1 INCREMENT BY 1;
CREATE TABLE PACKAGE
(
    ID              INTEGER DEFAULT NEXTVAL('S_PACKAGE') PRIMARY KEY,
    CODE            VARCHAR(100) NOT NULL UNIQUE,
    DIRECTORY       VARCHAR(60)  NOT NULL UNIQUE,
    TYPE            VARCHAR(20)  NOT NULL,
    PROJECT_ID      INTEGER,
    PACKAGE_FILE_ID INTEGER,
    DESCRIPTION     VARCHAR(125000),
    CONSTRAINT PACKAGE_PACKAGE_PROJECT_ID_FK
        FOREIGN KEY (PROJECT_ID) REFERENCES PROJECT
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_PACKAGE_FILE START WITH 1 INCREMENT BY 1;
CREATE TABLE PACKAGE_FILE
(
    ID          INTEGER   DEFAULT NEXTVAL('S_PACKAGE_FILE') PRIMARY KEY,
    FILE_NAME   VARCHAR(250)                          NOT NULL,
    PACKAGE_ID  INTEGER                               NOT NULL,
    VERSION     VARCHAR(100),
    DATE_UPLOAD TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    CHECKSUM    VARCHAR(50)                           NOT NULL,
    SIZE        BIGINT                                NOT NULL,
    UNIQUE (FILE_NAME, PACKAGE_ID),
    CONSTRAINT PACKAGE_FILE_PACKAGE_PACKAGE_ID_FK
        FOREIGN KEY (PACKAGE_ID) REFERENCES PACKAGE (ID)
            ON UPDATE CASCADE ON DELETE CASCADE
);

ALTER TABLE PACKAGE
    ADD CONSTRAINT PACKAGE_PACKAGE_FILE_PACKAGE_FILE_ID_FK
        FOREIGN KEY (PACKAGE_FILE_ID) REFERENCES PACKAGE_FILE (ID)
            ON UPDATE CASCADE ON DELETE SET NULL;

CREATE SEQUENCE S_PACKAGE_PARAMS START WITH 1 INCREMENT BY 1;
CREATE TABLE PACKAGE_PARAMS
(
    ID          INTEGER DEFAULT NEXTVAL('S_PACKAGE_PARAMS') PRIMARY KEY,
    PACKAGE_ID  INTEGER      NOT NULL,
    PARAM_NAME  VARCHAR(255) NOT NULL,
    PARAM_TYPE  VARCHAR(20)  NOT NULL,
    PARAM_VALUE VARCHAR(10000),
    UNIQUE (PACKAGE_ID, PARAM_NAME),
    CONSTRAINT PACKAGE_PARAMS_PACKAGE_PACKAGE_ID_FK
        FOREIGN KEY (PACKAGE_ID) REFERENCES PACKAGE
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_WORKFLOW START WITH 100 INCREMENT BY 1;
CREATE TABLE WORKFLOW
(
    ID               INTEGER     DEFAULT NEXTVAL('S_WORKFLOW') PRIMARY KEY,
    TITLE            VARCHAR(255)                NOT NULL UNIQUE,
    PROJECT_ID       INTEGER                     NOT NULL,
    SERVER_ID        INTEGER,
    ENABLED          BOOLEAN     DEFAULT TRUE    NOT NULL,
    STATUS           VARCHAR(60) DEFAULT 'EMPTY' NOT NULL,
    SUBSCRIBE_STATUS VARCHAR(40) DEFAULT 'NONE'  NOT NULL,
    START_DATE_TIME  TIMESTAMP,
    TYPE             VARCHAR(20)                 NOT NULL,
    DESCRIPTION      VARCHAR(125000),
    CONSTRAINT WORKFLOW_PROJECT_PROJECT_ID_FK
        FOREIGN KEY (PROJECT_ID) REFERENCES PROJECT (ID)
            ON UPDATE CASCADE,
    CONSTRAINT WORKFLOW_SERVER_SERVER_ID_FK
        FOREIGN KEY (SERVER_ID) REFERENCES SERVER (ID)
            ON UPDATE CASCADE
);

CREATE SEQUENCE S_WORKFLOW_PARAMS START WITH 1 INCREMENT BY 1;
CREATE TABLE WORKFLOW_PARAMS
(
    ID          BIGINT  DEFAULT NEXTVAL('S_WORKFLOW_PARAMS') PRIMARY KEY,
    WORKFLOW_ID INTEGER              NOT NULL,
    PARAM_TYPE  VARCHAR(20)          NOT NULL,
    PARAM_NAME  VARCHAR(255)         NOT NULL,
    PARAM_VALUE VARCHAR(10000),
    IS_ACTIVE   BOOLEAN DEFAULT TRUE NOT NULL,
    UNIQUE (WORKFLOW_ID, PARAM_TYPE, PARAM_NAME),
    CONSTRAINT OPERATOR_PARAMS_WORKFLOW_ID_FK
        FOREIGN KEY (WORKFLOW_ID) REFERENCES WORKFLOW
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_OPERATOR_STRUCTURE START WITH 100 INCREMENT BY 1;
CREATE TABLE OPERATOR_STRUCTURE
(
    ID             INTEGER     DEFAULT NEXTVAL('S_OPERATOR_STRUCTURE') PRIMARY KEY,
    PARENT_ID      INTEGER,
    WORKFLOW_ID    INTEGER                     NOT NULL,
    TYPE           VARCHAR(20) DEFAULT 'NONE'  NOT NULL,
    STATUS         VARCHAR(40) DEFAULT 'EMPTY' NOT NULL,
    CATEGORY       VARCHAR(20)                 NOT NULL,
    ORDER_CATEGORY SMALLINT                    NOT NULL,
    CONSTRAINT OPERATOR_STRUCTURE_PARENT_ID_FK
        FOREIGN KEY (PARENT_ID) REFERENCES OPERATOR_STRUCTURE (ID)
            ON UPDATE CASCADE ON DELETE CASCADE,
    CONSTRAINT OPERATOR_STRUCTURE_WORKFLOW_WORKFLOW_ID_FK
        FOREIGN KEY (WORKFLOW_ID) REFERENCES WORKFLOW (ID)
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE OPERATOR
(
    STRUCTURE_ID INTEGER PRIMARY KEY,
    TITLE        VARCHAR(1000)        NOT NULL,
    IS_ENABLED   BOOLEAN DEFAULT TRUE NOT NULL,
    CHARSET      VARCHAR(20),
    DESCRIPTION  VARCHAR(125000),
    CONSTRAINT OPERATOR_OPERATOR_STRUCTURE_STRUCTURE_ID_FK
        FOREIGN KEY (STRUCTURE_ID) REFERENCES OPERATOR_STRUCTURE (ID)
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE OPERATOR_BASH
(
    STRUCTURE_ID      INTEGER PRIMARY KEY,
    //PACKAGE_ID        INTEGER,
    COMMAND_DIRECTORY VARCHAR(1000),
    COMMAND           VARCHAR(10000),
    CONSTRAINT OPERATOR_BASH_OPERATOR_STRUCTURE_ID_FK
        FOREIGN KEY (STRUCTURE_ID) REFERENCES OPERATOR (STRUCTURE_ID)
            ON UPDATE CASCADE ON DELETE CASCADE
);


CREATE SEQUENCE S_OPERATOR_HISTORY START WITH 1 INCREMENT BY 1;
CREATE TABLE OPERATOR_HISTORY
(
    ID              BIGINT    DEFAULT NEXTVAL('S_OPERATOR_HISTORY') PRIMARY KEY,
    STRUCTURE_ID    INTEGER                               NOT NULL,
    SERVER_TITLE    VARCHAR(255)                          NOT NULL,
    SERVER_ID       INTEGER,
    STATUS          VARCHAR(60)                           NOT NULL,
    MODE            VARCHAR(60)                           NOT NULL,
    START_DATE_TIME TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL,
    END_DATE_TIME   TIMESTAMP,
    EXIT_VALUE      SMALLINT,
    ERROR           VARCHAR(20000),
    DESCRIPTION     VARCHAR(20000),
    LOG_FILE_STATUS VARCHAR(60)                           NOT NULL,
    CONSTRAINT OPERATOR_HISTORY_SERVERS_SERVER_ID_FK
        FOREIGN KEY (SERVER_ID) REFERENCES SERVER (ID)
            ON UPDATE CASCADE ON DELETE SET NULL,
    CONSTRAINT OPERATOR_HISTORY_OPERATOR_STRUCTURE_ID_FK
        FOREIGN KEY (STRUCTURE_ID) REFERENCES OPERATOR (STRUCTURE_ID)
            ON UPDATE CASCADE ON DELETE CASCADE
);

/*ALTER TABLE TASKS
    ADD CONSTRAINT TASKS_TASKS_HISTORY_TASK_HISTORY_ID_FK
        FOREIGN KEY (TASK_HISTORY_ID) REFERENCES TASKS_HISTORY (ID)
            ON UPDATE CASCADE ON DELETE SET NULL;*/

CREATE SEQUENCE S_OPERATOR_PARAMS START WITH 1 INCREMENT BY 1;
CREATE TABLE OPERATOR_PARAMS
(
    ID            BIGINT  DEFAULT NEXTVAL('S_OPERATOR_PARAMS') PRIMARY KEY,
    STRUCTURE_ID  INTEGER              NOT NULL,
    PARAM_TYPE    VARCHAR(20)          NOT NULL,
    PARAM_NAME    VARCHAR(255)         NOT NULL,
    PARAM_VALUE   VARCHAR(10000),
    DEFAULT_VALUE VARCHAR(10000),
    IS_ACTIVE     BOOLEAN DEFAULT TRUE NOT NULL,
    UNIQUE (STRUCTURE_ID, PARAM_TYPE, PARAM_NAME),
    CONSTRAINT OPERATOR_PARAMS_OPERATOR_STRUCTURE_ID_FK
        FOREIGN KEY (STRUCTURE_ID) REFERENCES OPERATOR
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE WORKFLOW_RULE
(
    WORKFLOW_ID INTEGER NOT NULL PRIMARY KEY,
    LONG_WORK   INTEGER,
    AUTO_STOP   INTEGER,
    DOUBLE_RUN  INTEGER,
    CONSTRAINT WORKFLOW_RULE_WORKFLOW_WORKFLOW_ID_FK
        FOREIGN KEY (WORKFLOW_ID) REFERENCES WORKFLOW
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE SEQUENCE S_PATTERN START WITH 1 INCREMENT BY 1;
CREATE TABLE PATTERN
(
    ID          INTEGER   DEFAULT NEXTVAL('S_PATTERN') PRIMARY KEY,
    TYPE        VARCHAR(20)                           NOT NULL,
    TITLE       VARCHAR(1000)                         NOT NULL,
    BODY        VARCHAR(10000)                        NOT NULL,
    DATE_CREATE TIMESTAMP DEFAULT CURRENT_TIMESTAMP() NOT NULL
);

CREATE SEQUENCE S_RUNTIME_SYSTEM_STATISTIC START WITH 1 INCREMENT BY 1;
CREATE TABLE RUNTIME_SYSTEM_STATISTIC
(
    ID               BIGINT DEFAULT NEXTVAL('S_RUNTIME_SYSTEM_STATISTIC') PRIMARY KEY,
    SERVER_ID        INTEGER   NOT NULL,
    INTERSPACE       INTEGER   NOT NULL,
    CPU_USED_PERCENT DOUBLE    NOT NULL,
    MEM_USED_PERCENT DOUBLE    NOT NULL,
    DATE_CREATE      TIMESTAMP NOT NULL,
    CONSTRAINT RUNTIME_SYSTEM_STATISTIC_SERVER_SERVER_ID_FK
        FOREIGN KEY (SERVER_ID) REFERENCES SERVER
            ON UPDATE CASCADE ON DELETE CASCADE
);

CREATE TABLE SETTINGS
(
    S_TYPE  VARCHAR(20)  NOT NULL,
    S_NAME  VARCHAR(255) NOT NULL,
    S_VALUE VARCHAR(1000),
    PRIMARY KEY (S_TYPE, S_NAME)
);